const bcrypt = require('bcrypt');
const jwt = require('jsonwebtoken');
const { Hotel, SuperAdmin } = require('../models');

/**
 * Hotel Admin Login
 */
const hotelLogin = async (req, res) => {
    try {
        const { email, password } = req.body;

        if (!email || !password) {
            return res.status(400).json({
                success: false,
                message: 'Email and password are required'
            });
        }

        // Find hotel by email
        const hotel = await Hotel.findOne({ where: { email } });

        if (!hotel) {
            return res.status(401).json({
                success: false,
                message: 'Invalid credentials'
            });
        }

        // Verify password
        const isPasswordValid = await bcrypt.compare(password, hotel.passwordHash);

        if (!isPasswordValid) {
            return res.status(401).json({
                success: false,
                message: 'Invalid credentials'
            });
        }

        // Check if account is active
        if (!hotel.isActive) {
            return res.status(403).json({
                success: false,
                message: 'Account is suspended. Please contact support.'
            });
        }

        // Check subscription
        const now = new Date();
        const subscriptionEnd = new Date(hotel.subscriptionEnd);

        if (subscriptionEnd < now) {
            return res.status(403).json({
                success: false,
                message: 'Subscription expired. Please renew to continue.',
                subscriptionExpired: true
            });
        }

        // Calculate remaining days
        const diffTime = subscriptionEnd.getTime() - now.getTime();
        const remainingDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));

        // Generate JWT token
        const token = jwt.sign(
            {
                hotelId: hotel.id,
                email: hotel.email,
                role: 'HOTEL_ADMIN'
            },
            process.env.JWT_SECRET,
            { expiresIn: process.env.JWT_EXPIRES_IN || '7d' }
        );

        res.json({
            success: true,
            message: 'Login successful',
            data: {
                token,
                hotel: {
                    id: hotel.id,
                    name: hotel.name,
                    email: hotel.email,
                    subscriptionEnd: hotel.subscriptionEnd,
                    remainingDays
                }
            }
        });
    } catch (error) {
        console.error('Hotel login error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error during login'
        });
    }
};

/**
 * Super Admin Login
 */
const superAdminLogin = async (req, res) => {
    try {
        const { email, password } = req.body;

        if (!email || !password) {
            return res.status(400).json({
                success: false,
                message: 'Email and password are required'
            });
        }

        // Find super admin
        const admin = await SuperAdmin.findOne({ where: { email } });

        if (!admin) {
            return res.status(401).json({
                success: false,
                message: 'Invalid credentials'
            });
        }

        // Verify password
        const isPasswordValid = await bcrypt.compare(password, admin.passwordHash);

        if (!isPasswordValid) {
            return res.status(401).json({
                success: false,
                message: 'Invalid credentials'
            });
        }

        // Generate JWT token
        const token = jwt.sign(
            {
                adminId: admin.id,
                email: admin.email,
                role: 'SUPER_ADMIN'
            },
            process.env.JWT_SECRET,
            { expiresIn: process.env.JWT_EXPIRES_IN || '7d' }
        );

        res.json({
            success: true,
            message: 'Login successful',
            data: {
                token,
                admin: {
                    id: admin.id,
                    email: admin.email,
                    role: 'SUPER_ADMIN'
                }
            }
        });
    } catch (error) {
        console.error('Super admin login error:', error);
        res.status(500).json({
            success: false,
            message: 'Server error during login'
        });
    }
};

module.exports = {
    hotelLogin,
    superAdminLogin
};
