-- ============================================
-- Hotel Ordering System - MySQL Database Schema
-- ============================================
-- Version: 1.0
-- Database: hotel_ordering_db
-- Character Set: utf8mb4
-- Collation: utf8mb4_unicode_ci
-- ============================================

-- Create database (if not exists)
CREATE DATABASE IF NOT EXISTS hotel_ordering_db 
CHARACTER SET utf8mb4 
COLLATE utf8mb4_unicode_ci;

USE hotel_ordering_db;

-- ============================================
-- Table: super_admins
-- Description: System administrators who manage hotels
-- ============================================
CREATE TABLE IF NOT EXISTS super_admins (
    id INT AUTO_INCREMENT PRIMARY KEY,
    email VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    createdAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    updatedAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: hotels
-- Description: Hotel accounts with subscription management
-- ============================================
CREATE TABLE IF NOT EXISTS hotels (
    id VARCHAR(36) PRIMARY KEY,
    name VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL UNIQUE,
    password_hash VARCHAR(255) NOT NULL,
    subscription_end DATETIME NOT NULL,
    is_active BOOLEAN NOT NULL DEFAULT TRUE,
    logo_url VARCHAR(500) DEFAULT 'https://via.placeholder.com/100',
    banner_url VARCHAR(500) DEFAULT 'https://via.placeholder.com/1200x400',
    whatsapp_number VARCHAR(20),
    phone VARCHAR(20),
    address TEXT,
    tagline VARCHAR(255),
    description TEXT,
    createdAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    updatedAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_subscription (subscription_end, is_active)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: categories
-- Description: Menu categories (scoped by hotel)
-- ============================================
CREATE TABLE IF NOT EXISTS categories (
    id INT AUTO_INCREMENT PRIMARY KEY,
    hotel_id VARCHAR(36) NOT NULL,
    name VARCHAR(100) NOT NULL,
    meal_times JSON NOT NULL,
    createdAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    updatedAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE,
    INDEX idx_hotel (hotel_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: menu_items
-- Description: Food items with images (scoped by hotel)
-- ============================================
CREATE TABLE IF NOT EXISTS menu_items (
    id VARCHAR(36) PRIMARY KEY,
    hotel_id VARCHAR(36) NOT NULL,
    category_id INT NOT NULL,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    price DECIMAL(10, 2) NOT NULL,
    offer_price DECIMAL(10, 2),
    image_url VARCHAR(500),
    meal_time ENUM('Breakfast', 'Lunch', 'Dinner') NOT NULL,
    dietary_type ENUM('Veg', 'Non-Veg', 'Vegan', 'None') DEFAULT 'None',
    is_special BOOLEAN DEFAULT FALSE,
    is_visible BOOLEAN DEFAULT TRUE,
    available_from TIME,
    available_to TIME,
    createdAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    updatedAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE,
    FOREIGN KEY (category_id) REFERENCES categories(id) ON DELETE CASCADE,
    INDEX idx_hotel (hotel_id),
    INDEX idx_category (category_id),
    INDEX idx_visibility (is_visible),
    INDEX idx_meal_time (meal_time)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Table: offers
-- Description: Promotional offers (scoped by hotel)
-- ============================================
CREATE TABLE IF NOT EXISTS offers (
    id VARCHAR(36) PRIMARY KEY,
    hotel_id VARCHAR(36) NOT NULL,
    name VARCHAR(255) NOT NULL,
    description TEXT,
    discount_type ENUM('Percentage', 'Fixed', 'Combo') NOT NULL,
    discount_value DECIMAL(10, 2) NOT NULL,
    start_date DATETIME NOT NULL,
    end_date DATETIME NOT NULL,
    target_meal_time ENUM('Breakfast', 'Lunch', 'Dinner'),
    target_category VARCHAR(100),
    createdAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    updatedAt DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (hotel_id) REFERENCES hotels(id) ON DELETE CASCADE,
    INDEX idx_hotel (hotel_id),
    INDEX idx_dates (start_date, end_date)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- ============================================
-- Insert Default Super Admin
-- Password: SuperAdmin@123 (hashed with bcrypt)
-- ============================================
-- Note: This is a placeholder. Run the initDatabase.js script instead
-- which will properly hash the password and create the super admin.

-- ============================================
-- Sample Queries
-- ============================================

-- Get all hotels with their item count
-- SELECT h.id, h.name, h.email, h.subscription_end, h.is_active,
--        COUNT(m.id) as item_count
-- FROM hotels h
-- LEFT JOIN menu_items m ON h.id = m.hotel_id
-- GROUP BY h.id;

-- Get menu items for a specific hotel with category info
-- SELECT m.*, c.name as category_name, c.meal_times
-- FROM menu_items m
-- JOIN categories c ON m.category_id = c.id
-- WHERE m.hotel_id = 'YOUR_HOTEL_ID'
-- AND m.is_visible = TRUE
-- ORDER BY m.createdAt DESC;

-- Get active offers for a hotel
-- SELECT * FROM offers
-- WHERE hotel_id = 'YOUR_HOTEL_ID'
-- AND start_date <= NOW()
-- AND end_date >= NOW()
-- ORDER BY createdAt DESC;

-- Check subscription status
-- SELECT id, name, email,
--        subscription_end,
--        is_active,
--        DATEDIFF(subscription_end, NOW()) as days_remaining,
--        CASE 
--            WHEN is_active = FALSE THEN 'Suspended'
--            WHEN subscription_end < NOW() THEN 'Expired'
--            WHEN DATEDIFF(subscription_end, NOW()) <= 7 THEN 'Expiring Soon'
--            ELSE 'Active'
--        END as status
-- FROM hotels;

-- ============================================
-- Database Statistics
-- ============================================

-- Total hotels
-- SELECT COUNT(*) as total_hotels FROM hotels;

-- Active subscriptions
-- SELECT COUNT(*) as active_hotels 
-- FROM hotels 
-- WHERE is_active = TRUE AND subscription_end > NOW();

-- Total menu items across all hotels
-- SELECT COUNT(*) as total_items FROM menu_items;

-- Items by meal time
-- SELECT meal_time, COUNT(*) as count 
-- FROM menu_items 
-- GROUP BY meal_time;

-- ============================================
-- Maintenance Queries
-- ============================================

-- Find expired subscriptions
-- SELECT id, name, email, subscription_end
-- FROM hotels
-- WHERE subscription_end < NOW()
-- AND is_active = TRUE;

-- Clean up old offers
-- DELETE FROM offers 
-- WHERE end_date < DATE_SUB(NOW(), INTERVAL 30 DAY);

-- ============================================
-- Backup Recommendation
-- ============================================
-- Run this command in terminal to backup:
-- mysqldump -u root -p hotel_ordering_db > backup_$(date +%Y%m%d).sql

-- To restore:
-- mysql -u root -p hotel_ordering_db < backup_20241129.sql

-- ============================================
-- End of Schema
-- ============================================
