const jwt = require('jsonwebtoken');
const { Hotel } = require('../models');

/**
 * Middleware to verify JWT token and attach user info to request
 */
const authenticateToken = async (req, res, next) => {
    try {
        const authHeader = req.headers['authorization'];
        const token = authHeader && authHeader.split(' ')[1]; // Bearer TOKEN

        if (!token) {
            return res.status(401).json({
                success: false,
                message: 'Access token required'
            });
        }

        // Verify token
        const decoded = jwt.verify(token, process.env.JWT_SECRET);

        // Attach user info to request
        req.user = decoded;

        next();
    } catch (error) {
        if (error.name === 'TokenExpiredError') {
            return res.status(401).json({
                success: false,
                message: 'Token expired'
            });
        }
        return res.status(403).json({
            success: false,
            message: 'Invalid token'
        });
    }
};

/**
 * Middleware to verify hotel admin token and check subscription
 */
const authenticateHotelAdmin = async (req, res, next) => {
    try {
        const authHeader = req.headers['authorization'];
        const token = authHeader && authHeader.split(' ')[1];

        if (!token) {
            return res.status(401).json({
                success: false,
                message: 'Access token required'
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);

        // Check if it's a hotel admin (not super admin)
        if (decoded.role !== 'HOTEL_ADMIN') {
            return res.status(403).json({
                success: false,
                message: 'Hotel admin access required'
            });
        }

        // Verify hotel exists and is active
        const hotel = await Hotel.findByPk(decoded.hotelId);

        if (!hotel) {
            return res.status(404).json({
                success: false,
                message: 'Hotel not found'
            });
        }

        if (!hotel.isActive) {
            return res.status(403).json({
                success: false,
                message: 'Hotel account is suspended'
            });
        }

        // Check subscription
        const now = new Date();
        const subscriptionEnd = new Date(hotel.subscriptionEnd);

        if (subscriptionEnd < now) {
            return res.status(403).json({
                success: false,
                message: 'Subscription expired. Please renew to continue.'
            });
        }

        // Attach hotel info to request
        req.user = decoded;
        req.hotel = hotel;

        next();
    } catch (error) {
        if (error.name === 'TokenExpiredError') {
            return res.status(401).json({
                success: false,
                message: 'Token expired'
            });
        }
        return res.status(403).json({
            success: false,
            message: 'Invalid token'
        });
    }
};

/**
 * Middleware to verify super admin token
 */
const authenticateSuperAdmin = async (req, res, next) => {
    try {
        const authHeader = req.headers['authorization'];
        const token = authHeader && authHeader.split(' ')[1];

        if (!token) {
            return res.status(401).json({
                success: false,
                message: 'Access token required'
            });
        }

        const decoded = jwt.verify(token, process.env.JWT_SECRET);

        if (decoded.role !== 'SUPER_ADMIN') {
            return res.status(403).json({
                success: false,
                message: 'Super admin access required'
            });
        }

        req.user = decoded;
        next();
    } catch (error) {
        if (error.name === 'TokenExpiredError') {
            return res.status(401).json({
                success: false,
                message: 'Token expired'
            });
        }
        return res.status(403).json({
            success: false,
            message: 'Invalid token'
        });
    }
};

module.exports = {
    authenticateToken,
    authenticateHotelAdmin,
    authenticateSuperAdmin
};
